package cz.frantovo.nekurak.ejb;

import cz.frantovo.nekurak.dao.UzivatelDAO;
import cz.frantovo.nekurak.dto.PozadavekNaRegistraciUzivatele;
import cz.frantovo.nekurak.dto.Uzivatel;
import cz.frantovo.nekurak.email.Adresa;
import cz.frantovo.nekurak.email.Postak;
import cz.frantovo.nekurak.util.Hash;
import cz.frantovo.nekurak.vyjimky.NecekanaVyjimka;
import cz.frantovo.nekurak.vyjimky.NekurakVyjimka;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;
import javax.ejb.Stateless;

/**
 *
 * @author fiki
 */
@Stateless
public class UzivatelEJB implements UzivatelRemote {

	@EJB
	private UzivatelDAO uzivatelDao;
	@EJB
	private Postak postak;
	private static final Logger log = Logger.getLogger(UzivatelEJB.class.getSimpleName());
	private Map<Locale, ResourceBundle> lokalizace = new HashMap<Locale, ResourceBundle>();

	/** Uživatele může zakládat kdokoli – uživatel se registruje sám. */
	@Override
	public void registrujUzivatele(PozadavekNaRegistraciUzivatele p) throws NekurakVyjimka {
		try {
			Uzivatel u = p.getUzivatel();
			/** Heslo musíme před uložení zahashovat, aby se uživatel mohl přihlásit. */
			u.setHeslo(Hash.hashuj(u.getHeslo()));
			uzivatelDao.zaloz(u);
			posliRegistracniEmail(p);
		} catch (NoSuchAlgorithmException e) {
			/** Nemělo by nastat */
			throw new NecekanaVyjimka("Neexistující hashovací algoritmus.", e);
		} catch (UnsupportedEncodingException e) {
			/** Nemělo by nastat */
			throw new NecekanaVyjimka("Nepodporované kódování znaků.", e);
		}
	}

	private ResourceBundle getLokalizace(Locale jazyk) {
		if (jazyk == null) {
			jazyk = Locale.getDefault();
		}

		if (lokalizace.get(jazyk) == null) {
			lokalizace.put(jazyk, ResourceBundle.getBundle("cz.frantovo.nekurak.preklady", jazyk));
		}

		return lokalizace.get(jazyk);
	}

	private void posliRegistracniEmail(PozadavekNaRegistraciUzivatele p) {
		try {
			String email = p.getEmail();
			if (Postak.isAdresaOk(email)) {
				ResourceBundle rb = getLokalizace(p.getJazyk());
				String jmeno = p.getJmeno() + " " + p.getPrijmeni();
				String predmet = rb.getString("registrace.email.predmet");
				Object[] hodnotyTextu = {jmeno, p.getPrezdivka()};
				String text = new MessageFormat(rb.getString("registrace.email.text"), p.getJazyk()).format(hodnotyTextu);
				postak.odesliZpravu(new Adresa(email, jmeno), null, predmet, text);
			} else {
				log.log(Level.WARNING, "E-mailová adresa buď chybí nebo je chybná → neodesíláme registrační e-mail. Uživatel: {0}", p.getPrezdivka());
			}
		} catch (NekurakVyjimka e) {
			/** e-mail je jen informační → pokud se ho nepodaří odeslat, nemusíme stornovat registraci. */
			log.log(Level.WARNING, "Chyba při posílání registračního e-mailu. Uživatel: " + p.getPrezdivka(), e);
		}
	}
}
